
import { Injectable } from '@angular/core';
import { BehaviorSubject, Subject, interval, Subscription } from 'rxjs';
import { TcpSocket, DataEncoding } from 'capacitor-tcp-socket';

/** Build firmware command lines (pad to 5 params, end with \n) */
function buildCommand(cmd: string, params: number[] = []): string {
  const filled = [...params, 0, 0, 0, 0, 0].slice(0, 5);
  return `${cmd}#${filled.join('#')}\n`;
}

export type ModuleFlag = 1 | 2 | 0; // 1: MATRIX, 2: SONAR

@Injectable({ providedIn: 'root' })
export class CarService {
  // Connection
  private clientId: number | null = null;
  readonly connected$ = new BehaviorSubject<boolean>(false);
  readonly status$ = new Subject<string>();
  readonly lastRaw$ = new Subject<string>();

  // Data
  readonly moduleFlag$ = new BehaviorSubject<ModuleFlag>(0);
  readonly batteryVoltage$ = new BehaviorSubject<number | null>(null);

  // Reader
  private readerSub?: Subscription;

  async connect(hostPort = '192.168.4.1:4002') {
    try {
      if (this.clientId !== null) await this.disconnect();
      const [ipAddress, portStr] = hostPort.split(':');
      const port = Number(portStr || 4002);
      const res = await TcpSocket.connect({ ipAddress, port });
      this.clientId = res.client;
      this.connected$.next(true);
      this.status$.next(`Connected to ${ipAddress}:${port}`);
      this.startReader();
    } catch (err: any) {
      this.status$.next('Connect failed');
      this.connected$.next(false);
      throw err;
    }
  }

  private async disconnect() {
    try {
      if (this.clientId !== null) {
        await TcpSocket.disconnect({ client: this.clientId });
      }
    } finally {
      this.clientId = null;
      this.connected$.next(false);
      this.stopReader();
    }
  }

  private startReader() {
    this.stopReader();
    // Poll-read every 80 ms; expect up to 1024 bytes
    this.readerSub = interval(80).subscribe(async () => {
      if (this.clientId === null) return;
      try {
        const out = await TcpSocket.read({ client: this.clientId, expectLen: 1024, encoding: DataEncoding.UTF8 });
        if (!out?.result) return;
        const text = String(out.result);
        this.lastRaw$.next(text);
        text.split('\n').filter(Boolean).forEach(line => this.parseLine(line.trim()));
      } catch {
        // ignore timeouts / transient errors
      }
    });
  }
  private stopReader() {
    this.readerSub?.unsubscribe();
    this.readerSub = undefined;
  }

  private async sendRaw(payload: string) {
    if (this.clientId === null) return;
    await TcpSocket.send({ client: this.clientId, data: payload, encoding: DataEncoding.UTF8 });
  }

  private parseLine(line: string) {
    const [tag, ...parts] = line.split('#');
    if (tag === 'A' && parts[0]) {
      const flag = Number(parts[0]) as ModuleFlag;
      if (flag === 1 || flag === 2) this.moduleFlag$.next(flag);
    }
    if (tag === 'P' && parts[0]) {
      const v = Number(parts[0]);
      if (!Number.isNaN(v)) this.batteryVoltage$.next(v);
    }
  }

  // === Commands ===
  setLedMode(mode: number) {
    this.sendRaw(buildCommand('D', [mode]));
  }
  setLedColor(mask: number, r: number, g: number, b: number) {
    this.sendRaw(buildCommand('L', [mask, r, g, b]));
  }
  setMatrixMode(mode: number) {
    this.sendRaw(buildCommand('T', [mode]));
  }
  setCarMode(mode: number) {
    this.sendRaw(buildCommand('C', [mode]));
  }
  setServo(angle: number) {
    this.sendRaw(buildCommand('S', [Math.round(angle)]));
  }
  buzzer(freq: number) {
    this.sendRaw(buildCommand('B', [Math.round(freq)]));
  }
  getPower() {
    this.sendRaw(buildCommand('P', [0]));
  }
  driveDual(angleL: number, magL: number, angleR: number, magR: number, rotate = false) {
    const aL = Math.max(0, Math.min(180, Math.round(angleL)));
    const aR = Math.max(0, Math.min(180, Math.round(angleR)));
    const mL = Math.max(0, Math.min(100, Math.round(magL)));
    const mR = Math.max(0, Math.min(100, Math.round(magR)));
    const cmd = rotate ? 'O' : 'N';
    this.sendRaw(buildCommand(cmd, [aL, mL, aR, mR]));
  }
}
