import { Component } from '@angular/core';
import {
  IonApp,
  IonHeader,
  IonToolbar,
  IonTitle,
  IonContent,
  IonItem,
  IonLabel,
  IonInput,
  IonButton,
  IonList,
  IonToggle,
  IonRange,
  IonGrid,
  IonRow,
  IonCol,
  IonSelect,
  IonSelectOption,
  IonSegment,
  IonSegmentButton,
  IonBadge,
  IonChip,
  IonIcon,
  IonCard,
  IonCardHeader,
  IonCardTitle,
  IonCardContent,
  IonButtons,
} from '@ionic/angular/standalone';
import { CommonModule } from '@angular/common';
import { FormsModule } from '@angular/forms';
import { CarService } from '../../services/car.service';
import { addIcons } from 'ionicons';
import {
  play,
  stop,
  batteryCharging,
  scanCircle,
  colorPalette,
} from 'ionicons/icons';

addIcons({ play, stop, batteryCharging, scanCircle, colorPalette });

@Component({
  standalone: true,
  selector: 'app-control',
  imports: [
    IonButtons,
    CommonModule,
    FormsModule,
    IonApp,
    IonHeader,
    IonToolbar,
    IonTitle,
    IonContent,
    IonItem,
    IonLabel,
    IonInput,
    IonButton,
    IonToggle,
    IonRange,
    IonGrid,
    IonRow,
    IonCol,
    IonSelect,
    IonSelectOption,
    IonSegment,
    IonSegmentButton,
    IonBadge,
    IonChip,
    IonIcon,
    IonCard,
    IonCardHeader,
    IonCardTitle,
    IonCardContent,
  ],
  templateUrl: './control.page.html',
  styleUrls: ['./control.page.scss'],
})
export class ControlPage {
  // Target
  target = '192.168.8.128:4002';
  connected = false;
  moduleFlag: 0 | 1 | 2 = 0;
  battery: number | null = null;

  // LEDs
  ledMode = 1;
  ledMask = 255;
  color = { r: 0, g: 0, b: 100 };

  // Matrix
  matrixMode = 1;

  // Car mode
  carMode = 0;

  // Servo & Buzzer
  servoAngle = 90;
  buzzerFreq = 0;

  // Manual drive
  useRotate = false;
  leftAngle = 90;
  leftMag = 0;
  rightAngle = 90;
  rightMag = 0;

  private lastSend = 0;
  private sendIntervalMs = 50;

  constructor(public car: CarService) {
    this.car.connected$.subscribe((v) => (this.connected = v));
    this.car.moduleFlag$.subscribe((f) => (this.moduleFlag = f));
    this.car.batteryVoltage$.subscribe((v) => (this.battery = v));
  }

  connect() {
    this.car.connect(this.target);
  }

  applyLedMode() {
    this.car.setLedMode(this.ledMode);
  }
  applyLedColor() {
    this.car.setLedColor(
      this.ledMask,
      this.color.r,
      this.color.g,
      this.color.b
    );
  }
  applyMatrix() {
    this.car.setMatrixMode(this.matrixMode);
  }
  applyCarMode() {
    this.car.setCarMode(this.carMode);
  }
  applyServo() {
    this.car.setServo(this.servoAngle);
  }
  playBuzzer() {
    this.car.buzzer(this.buzzerFreq);
  }
  stopBuzzer() {
    this.car.buzzer(0);
  }
  readBattery() {
    this.car.getPower();
  }

  toggleLedBit(i: number) {
    const bit = 1 << i;
    this.ledMask = (this.ledMask ^ bit) >>> 0;
  }

  onStickMove(which: 'L' | 'R', ev: PointerEvent) {
    const el = ev.currentTarget as HTMLElement;
    const rect = el.getBoundingClientRect();
    const cx = rect.left + rect.width / 2;
    const cy = rect.top + rect.height / 2;
    const dx = ev.clientX - cx;
    const dy = cy - ev.clientY;
    const radius = rect.width / 2;
    const mag = Math.min(100, Math.round((Math.hypot(dx, dy) / radius) * 100));
    const angleRad = Math.atan2(dx, dy);
    let angleDeg = Math.round(((angleRad * 180) / Math.PI + 360) % 360);
    if (angleDeg > 180) angleDeg = 360 - angleDeg;

    if (which === 'L') {
      this.leftAngle = angleDeg;
      this.leftMag = mag;
    } else {
      this.rightAngle = angleDeg;
      this.rightMag = mag;
    }
    const now = performance.now();
    if (now - this.lastSend > this.sendIntervalMs) {
      this.lastSend = now;
      this.car.driveDual(
        this.leftAngle,
        this.leftMag,
        this.rightAngle,
        this.rightMag,
        this.useRotate
      );
    }
  }

  onStickEnd(which: 'L' | 'R') {
    if (which === 'L') {
      this.leftMag = 0;
    } else {
      this.rightMag = 0;
    }
    this.car.driveDual(
      this.leftAngle,
      this.leftMag,
      this.rightAngle,
      this.rightMag,
      this.useRotate
    );
  }

  setAllLeds(on: boolean) {
    this.ledMask = on ? 255 : 0;
  }

  ledIndices = [0, 1, 2, 3, 4, 5, 6, 7];

  isLedOn(i: number): boolean {
    return (this.ledMask & (1 << i)) !== 0;
  }
}
